
#ifndef _15_CRYS_DH_h_H
#define _15_CRYS_DH_h_H

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */



#ifdef __cplusplus
extern "C"
{
#endif

/*
 *  Object name     :  CRYS_DH.h
 */

  /*
   *  Object %name    : %CRYS_DH.h
   *  State           :  %state%
   *  Creation date   :  Wed Jan 26 16:54:02 2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief This module defines the API that supports Diffie-Hellman key exchange 
   *  as defined in PKCS#3 and ANSI9.42 v1.4 (Revised from November 1,1993)
   *
   *  \version CRYS_DH.h#1:incl:15
   *  \author ohads
   *  \remarks Copyright (C) 2005 by Discretix Technologies Ltd.
   *           All Rights reserved
   */




/************************ Defines ******************************/
/*The following defines are for the readability of the user*/
#define CRYS_DHPrimeData_t CRYS_RSAPrimeData_t
#define CRYS_DHUserPubKey_t CRYS_RSAUserPubKey_t
/************************ Enums ********************************/

typedef enum
{
   CRYS_DH_PKCS3_mode  = 0,
   CRYS_DH_ANSI_X942_mode = 1,
   CRYS_DH_NumOfModes,

   CRYS_DH_OpModeLast    = 0x7FFFFFFF,

}CRYS_DH_OpMode_t;   


typedef enum
{
   CRYS_DH_HASH_SHA1_mode  = 0,
   CRYS_DH_HASH_NumOfModes,

   CRYS_DH_HASH_OpModeLast= 0x7FFFFFFF,

}CRYS_DH_HASH_OpMode_t;   

typedef enum
{
   CRYS_DH_ASN1_Der_mode  = 0,
   CRYS_DH_Concat_Der_mode  = 1,   
   CRYS_DH_der_NumOfModes,

   CRYS_DH_DerivationFunc_ModeLast= 0x7FFFFFFF,

}CRYS_DH_DerivationFunc_Mode;   

/************************ Typedefs  ****************************/


/************************ Structs  ******************************/


/************************ Public Variables **********************/


/************************ Public Functions ******************************/

/**
 * @brief 
 	_DX_DH_GeneratePubPrv has 2 functionalities:
 	1. randomly generates the Client private key 
 	2. compute the Client public key which is 
 		ClientPub = Generator ^ Prv mod Prime 
 		
 	Note: All buffers arguments are represented in Big-Endian
 	
  @param[in] Generator_ptr 		- Pointer to the Generator octet string
  @param[in] GeneratorSize 		- Size of the Generator String (in bytes)
  @param[in] Prime_ptr 			- Pointer to the Prime octet string
  @param[in] PrimeSize 			- Size of the Prime string (in bytes)
  @param[in] L 					- Relevant only for PKCS#3, the private value length in bits
  								  If L != 0 then - force the private key to be 2^(L-1) <= Prv < 2^l
  				 					if L 0 then: 0< Prv < P-1
  				 				  In ANSI X9.42 L is irrelevant  
  @param[in] Q_ptr 				- Relevant only for ANSI X9.42 - Pointer to the Q octet string
  									1<= Prv <= q-1   or   1< Prv < q-1
  @param[in] QSize	 			- Relevant only for ANSI X9.42 - Size of the Q string (in bytes)  				 				  
  @param[in] DH_mode			- enum declaring weather this is PKCS#3 or ANSI X9.42
  
  @param[in] UserPubKey_ptr		- a pointer to the publick key structure. used for the Exp operation function
  									the struct doesn't need to be initialized
  @param[in] PrimeData_ptr 		- a pointer to a structure containing internal buffers  
                                    the struct doesn't need to be initialized
  @param[out] ClientPrvKey_ptr  - Pointer to the Private key octet string - 
  							  	  In PKCS#3 This buffer should be at least the following size
	  							  	  2^L - 1 - if L is provided (size of buffer*8 >= L).
	  							  	  P-1	  - if L is DX_NULL (size of buffer >= PrimeSize).
								  In X9.42 size of buffer >= QSize.	  							  	  
  @param[in/out] ClientPrvKeySize_ptr - The user should provide the size of the buffer indicated by ClientPrvKey_ptr
                                        The function returns the actual size in bytes of the Private key Size
  @param[out] ClientPub_ptr 	- Pointer to the Public key octet string
  						  		  This Buffer should be at least PrimeSize bytes
  							  
  @param[in/out] ClientPubSize_ptr -    The user should provide the size of the buffer indicated by ClientPub_ptr
                                        The function returns the actual size in bytes of the client public buffer
 
  @return CRYSError_t - On success CRYS_OK is returned, on failure an ERROR as defined CRYS_DH_error.h:
                        CRYS_DH_INVALID_ARGUMENT_POINTER_ERROR;      
   						CRYS_DH_INVALID_ARGUMENT_SIZE_ERROR;
    					CRYS_DH_INVALID_ARGUMENT_OPERATION_MODE_ERROR;   						
						CRYS_DH_ARGUMENT_GENERATOR_SMALLER_THAN_ZERO_ERROR;
						CRYS_DH_ARGUMENT_PRIME_SMALLER_THAN_GENERATOR_ERROR;						
						CRYS_DH_INVALID_L_ARGUMENT_ERROR;
						CRYS_DH_ARGUMENT_PRV_SIZE_ERROR;						
    					CRYS_DH_INVALID_ARGUMENT_OPERATION_MODE_ERROR;
 */
CIMPORT_C CRYSError_t _DX_DH_GeneratePubPrv(DxUint8_t *Generator_ptr,
				                    DxUint16_t GeneratorSize,
				                    DxUint8_t *Prime_ptr,
				                    DxUint16_t PrimeSize,
				                    DxUint16_t L,
				                    DxUint8_t *Q_ptr,
				                    DxUint16_t QSize,
				                    CRYS_DH_OpMode_t DH_mode,
   				                    CRYS_DHUserPubKey_t *UserPubKey_ptr,
									CRYS_DHPrimeData_t  *PrimeData_ptr,   				                    
				                    DxUint8_t *ClientPrvKey_ptr,
				                    DxUint16_t *ClientPrvKeySize_ptr,
                                    DxUint8_t *ClientPub1_ptr,
				                    DxUint16_t *ClientPubSize_ptr);



/*Q is irrelevant for PKCS#3 */
#define CRYS_DH_PKCS3_GeneratePubPrv(Generator_ptr,GeneratorSize,\
									Prime_ptr,PrimeSize,\
									L,\
									UserPubKey_ptr,PrimeData_ptr,\
									ClientPrvKey_ptr,ClientPrvKeySize_ptr,\
									ClientPub_ptr,ClientPubSize_ptr)\
		_DX_DH_GeneratePubPrv(Generator_ptr,GeneratorSize,\
								Prime_ptr,PrimeSize,\
								L,\
								(DxUint8_t *)DX_NULL,(DxUint16_t)0,\
								CRYS_DH_PKCS3_mode,\
								UserPubKey_ptr,PrimeData_ptr,\
								ClientPrvKey_ptr,ClientPrvKeySize_ptr,\
								ClientPub_ptr,ClientPubSize_ptr)

/*L is irrelevant for ANSI X9.42*/
#define CRYS_DH_ANSI_X942_GeneratePubPrv(Generator_ptr,GeneratorSize,Prime_ptr,PrimeSize,\
									Q_ptr,QSize,\
									UserPubKey_ptr,PrimeData_ptr,\
									ClientPrvKey_ptr,ClientPrvKeySize_ptr,\
									ClientPub_ptr,ClientPubSize_ptr)\
		_DX_DH_GeneratePubPrv(Generator_ptr,GeneratorSize,\
								Prime_ptr,PrimeSize,\
								(DxUint16_t)0,\
								Q_ptr,QSize,\
								CRYS_DH_ANSI_X942_mode,\
								UserPubKey_ptr,PrimeData_ptr,\
								ClientPrvKey_ptr,ClientPrvKeySize_ptr,\
								ClientPub_ptr,ClientPubSize_ptr)

/******************************************************************************************/
/**
 * @brief CRYS_DH_PKCS3_GetSecretKey computes the shared secret key in the following computation:
 	SecretKey = ServerPubKey ^ ClientPrvKey mod Prime
 
  	Note: All buffers arguments are represented in Big-Endian
 
  @param[in] ClientPrvKey_ptr 	- Pointer to the Private key octet string - 
  @param[in] ClientPrvKeySize  	- The Private key Size (in bytes)
  @param[in] ServerPubKey_ptr   - Pointer to the Server public key octet string - 
  @param[in] ServerPubKeySize   - The Server Public key Size (in bytes)
  @param[in] Prime_ptr 			- Pointer to the Prime octet string
  @param[in] PrimeSize 			- Size of the Prime string
  @param[in] UserPubKey_ptr		- a pointer to the publick key structure. used for the Exp operation function
                                  the struct doesn't need to be initialized
  @param[in] PrimeData_ptr 		- a pointer to a structure containing internal buffers
                                  the struct doesn't need to be initialized    
  @param[out] SecretKey_ptr 	- Pointer to the secret key octet string.
  							  	  This buffer should be at least PrimeSize Bytes
  							  
  @param[in/out] SecretKeySize_ptr - The user should provide the actual size in bytes of the buffer indicated by SecretKey_ptr
                                    The function will return the actual size in bytes of the output secret key

  @return CRYSError_t - On success CRYS_OK is returned, on failure an ERROR as defined CRYS_DH_error.h:
					    CRYS_DH_INVALID_ARGUMENT_POINTER_ERROR;      
   					    CRYS_DH_INVALID_ARGUMENT_SIZE_ERROR;
					    CRYS_DH_SECRET_KEY_SIZE_OUTPUT_ERROR;

*/
CIMPORT_C CRYSError_t CRYS_DH_PKCS3_GetSecretKey(DxUint8_t *ClientPrvKey_ptr,
				                    DxUint16_t ClientPrvKeySize,
 				                    DxUint8_t *ServerPubKey_ptr,
				                    DxUint16_t ServerPubKeySize,				                    
				                    DxUint8_t *Prime_ptr,
				                    DxUint16_t PrimeSize,
                                    CRYS_DHUserPubKey_t *UserPubKey_ptr,
                                    CRYS_DHPrimeData_t  *PrimeData_ptr, 				                    
				                    DxUint8_t *SecretKey_ptr,
				                    DxUint16_t *SecretKeySize_ptr);


/******************************************************************************************/
/**
 * @brief _DX_DH_KeyDerivationFunc specifies the key derivation function, either based on ASN.1 DER encoding
 			or based on the concatanation of fields as specified in ANSI X9.42-2001.
 			The purpose of this function is to derive a key data from the shared secret value.
 			
	The actual APIs that will be used by the user are:
		CRYS_DH_ASN1_KeyDerivation_SHA1Func 			
		CRYS_DH_Concat_KeyDerivation_SHA1Func
 			
  \note The length in Bytes of the hash result buffer is denoted by "hashlen".
  \note The standard states that optional "other info" may be passed as input to
        the derivation function. This implementation does not support this option.
  \note All buffers arguments are represented in Big-Endian format.
 
  @param[in] ZZSecret_ptr 	- A pointer to shared secret key octet string 
  @param[in] ZZSecretSize  	- The shared secret key Size, in bytes
  @param[in] hashFunc 		- The hash function to be used. The hash function output must be at least 160 bits.
  @param[in] KeyLenInBits	- The size in Bits of the keying data to be generated - 
  								KeyLenInBits < hashlen*(2^32-1) - in our implementation the size should not 
  								be larger than the maximum input size to the Hash function
  @param[in] derivation_mode - Specifies whether we use an ASN.1-based derivation function 
                               or a derivation based on concatenation.
  @param[out] KeyingData_ptr - A pointer to the keying data derived from the secret key, of length KeyLenInBits
  							  
  @return CRYSError_t - On success the value CRYS_OK is returned, and on failure an ERROR as defined in CRYS_DH_error.h:
  						CRYS_DH_INVALID_ARGUMENT_POINTER_ERROR;
   						CRYS_DH_INVALID_ARGUMENT_SIZE_ERROR;
   						CRYS_DH_INVALID_ARGUMENT_HASH_MODE_ERROR;

*/
CIMPORT_C CRYSError_t _DX_DH_KeyDerivationFunc(DxUint8_t *ZZSecret_ptr,
				                      DxUint16_t ZZSecretSize,
				  				      CRYS_DH_HASH_OpMode_t hashFunc,				                    
				                      DxUint32_t KeyLenInBits,				                    
				                      DxUint8_t *KeyingData_ptr,
				                      CRYS_DH_DerivationFunc_Mode derivation_mode);
				                    

				                    
#define CRYS_DH_ASN1_KeyDerivation_SHA1Func(ZZSecret_ptr,ZZSecretSize,KeyLenInBits,KeyingData_ptr)\
		_DX_KeyDerivationFunc(ZZSecret_ptr,ZZSecretSize,CRYS_DH_HASH_SHA1_mode,KeyLenInBits,KeyingData_ptr,CRYS_DH_ASN1_Der_mode)
				                    

#define CRYS_DH_Concat_KeyDerivation_SHA1Func(ZZSecret_ptr,ZZSecretSize,KeyLenInBits,KeyingData_ptr)\
		_DX_KeyDerivationFunc(ZZSecret_ptr,ZZSecretSize,CRYS_DH_HASH_SHA1_mode,KeyLenInBits,KeyingData_ptr,CRYS_DH_Concat_Der_mode)


/******************************************************************************************/
/**
 * @brief _DX_DH_X942_GetSecretData computes the shared secret key as follows:
 		1. OPTIONAL - validate the correctness of the argument keys
 		2. SecretKey = ServerPubKey ^ ClientPrvKey mod Prime
 		3. Use of Derivation function to derive a data key from the secret key 
 
 	The actual APIs that will be used by the user are:
 	CRYS_DH_X942_GetSecret_ASN1_2_Data
 	CRYS_DH_X942_GetSecret_Concat_2_Data
 
  	Note: All buffers arguments are represented in Big-Endian format

  @param[in] UserPubKey_ptr		- A pointer to the public key data structure. Not initialized.
  @param[in] PrimeData_ptr 		- A pointer to a CRYS_RSAPrimeData_t structure 
   								  that is used for the Exp operation
  @param[in] hashFunc			- The hash function to be used
  @param[in] ClientPrvKey_ptr 	- A pointer to the Private key octet string 
  @param[in] ClientPrvKeySize  	- The Private key size, in bytes
  @param[in] ServerPubKey_ptr   - A pointer to the Server public key octet string - 
  @param[in] ServerPubKeySize   - The Server Public key size, in bytes
  @param[in] Prime_ptr 			- A pointer to the Prime octet string
  @param[in] PrimeSize 			- The size of the Prime string
  @param[in] DerFunc_mode 		- The type of function to derive the secret key to the key data.
  								  We use ASN.1-based function or Hash concatenation function mode.
  @param[out] SecretKeyData_ptr	- A pointer to the secret key octet string. 
				  				  This buffer should be at least PrimeSize bytes.
  @param[in] SecretKeyDataSizeBitsNeeded - Specifies the derived Secret Key data size needed in Bits. This value converted to bytes 
                                  cannot be larger than PrimeSize (required by implementation)

  @return CRYSError_t - On success CRYS_OK is returned, on failure an ERROR as defined CRYS_DH_error.h:
						CRYS_DH_SECRET_KEY_SIZE_NEEDED_ERROR;
						CRYS_DH_SECRET_KEY_SIZE_NEEDED_BIGGER_THAN_PRIME_SIZE;
*/
CIMPORT_C CRYSError_t _DX_DH_X942_GetSecretData(CRYS_DHUserPubKey_t *UserPubKey_ptr,
 									CRYS_DHPrimeData_t  *PrimeData_ptr, 
				  				    CRYS_DH_HASH_OpMode_t hashFunc,		
									DxUint8_t *ClientPrvKey_ptr,
				                    DxUint16_t ClientPrvKeySize,
 				                    DxUint8_t *ServerPubKey_ptr,
				                    DxUint16_t ServerPubKeySize,				                    
				                    DxUint8_t *Prime_ptr,
				                    DxUint16_t PrimeSize,
				                    CRYS_DH_DerivationFunc_Mode DerFunc_mode,
				                    DxUint8_t *SecretKeyData_ptr,
				                    DxUint16_t SecretKeyDataSizeBitsNeeded);

/**
   \brief CRYS_DH_X942_GetSecret_ASN1_2_Data implements the DH ANSI X9.42 standard. It derives a secret key 
   			using the Derivation function based on ASN1
*/
#define CRYS_DH_X942_GetSecret_ASN1_2_Data(UserPubKey_ptr,PrimeData_ptr,hashFunc,ClientPrvKey_ptr,ClientPrvKeySize,ServerPubKey_ptr,ServerPubKeySize,Prime_ptr,PrimeSize,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)\
	_DX_DH_X942_GetSecretData(UserPubKey_ptr,PrimeData_ptr,hashFunc,ClientPrvKey_ptr,ClientPrvKeySize,ServerPubKey_ptr,ServerPubKeySize,Prime_ptr,PrimeSize,CRYS_DH_ASN1_Der_mode,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)
/**
   \brief CRYS_DH_X942_GetSecret_Concat_2_Data implements the DH ANSI X9.42 standard. It derives a secret key 
   			using the Derivation function based on concatenation of SHA1 hash 
*/				                    
#define CRYS_DH_X942_GetSecret_Concat_2_Data(UserPubKey_ptr,PrimeData_ptr,hashFunc,ClientPrvKey_ptr,ClientPrvKeySize,ServerPubKey_ptr,ServerPubKeySize,Prime_ptr,PrimeSize,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)\
	_DX_DH_X942_GetSecretData(UserPubKey_ptr,PrimeData_ptr,hashFunc,ClientPrvKey_ptr,ClientPrvKeySize,ServerPubKey_ptr,ServerPubKeySize,Prime_ptr,PrimeSize,CRYS_DH_Concat_Der_mode,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)
				                    
/******************************************************************************************/
/**
 * @brief _DX_DH_X942_Hybrid_GetSecretData computes the shared secret key as follows:
 		1. OPTIONAL - validate the correctness of the argument keys
 		2. SecretKey1 = ServerPubKey1 ^ ClientPrvKey1 mod Prime
 		3. SecretKey2 = ServerPubKey2 ^ ClientPrvKey2 mod Prime
 		3. Use of Derivation function to derive a data key from the 2 secret keys 
 
 	The actual APIs that will be used by the user are:
 	CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data
 	CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data
 
  	Note: All buffers arguments are represented in Big-Endian format.
 
  @param[in] UserPubKey_ptr		- A pointer to the public key data structure. Not initialized.
  @param[in] PrimeData_ptr 		- A pointer to a CRYS_RSAPrimeData_t structure 
                                  that is used for the Exp operation
  @param[in] hashFunc			- The hash function to be used
  @param[in] ClientPrvKey_ptr1 	- A pointer to the First Private key octet string number 
  @param[in] ClientPrvKeySize1 	- The First Private key Size, in bytes
  @param[in] ClientPrvKey_ptr2 	- A pointer to the Second Private key octet string
  @param[in] ClientPrvKeySize2 	- The Second Private key Size, in bytes
  @param[in] ServerPubKey_ptr1  - A pointer to the First Server public key octet string
  @param[in] ServerPubKeySize1  - The First Server Public key Size, in bytes
  @param[in] ServerPubKey_ptr2  - A pointer to the Second Server public key octet string
  @param[in] ServerPubKeySize2  - The Second Server Public key Size, in bytes
  @param[in] Prime_ptr 			- A pointer to the Prime octet string
  @param[in] PrimeSize 			- The size of the Prime string
  @param[in] DerFunc_mode 		- The type of function to derive the secret key to the key data.
                                  We use an ASN.1-based function or a Hash concatenation function mode.
  @param[out] SecretKeyData_ptr - A pointer to the secret key octet string.
			  	                  This buffer should be at least 2*PrimeSize bytes.
  							  
  @param[in] SecretKeyDataSizeBitsNeeded - Specifies the derived Secret Key data size needed in Bits. This value converted to bytes 
                                  cannot be larger than PrimeSize (required by implementation)

 @return CRYSError_t - On success the value CRYS_OK is returned, and on failure an ERROR as defined in CRYS_DH_error.h
						CRYS_DH_SECRET_KEY_SIZE_NEEDED_ERROR;
						CRYS_DH_SECRET_KEY_SIZE_NEEDED_BIGGER_THAN_PRIME_SIZE;
						CRYS_DH_X942_HYBRID_SIZE1_BUFFER_ERROR;
                        CRYS_DH_INVALID_ARGUMENT_POINTER_ERROR;      
   						CRYS_DH_INVALID_ARGUMENT_SIZE_ERROR;
    					CRYS_DH_INVALID_ARGUMENT_OPERATION_MODE_ERROR;   						
						CRYS_DH_ARGUMENT_GENERATOR_SMALLER_THAN_ZERO_ERROR;
						CRYS_DH_ARGUMENT_PRIME_SMALLER_THAN_GENERATOR_ERROR;						
						CRYS_DH_INVALID_L_ARGUMENT_ERROR;
						CRYS_DH_ARGUMENT_PRV_SIZE_ERROR;						
    					CRYS_DH_INVALID_ARGUMENT_OPERATION_MODE_ERROR;
*/
CIMPORT_C CRYSError_t _DX_DH_X942_Hybrid_GetSecretData(CRYS_DHUserPubKey_t *UserPubKey_ptr,
									CRYS_DHPrimeData_t  *PrimeData_ptr, 
				  				    CRYS_DH_HASH_OpMode_t hashFunc,		
 									DxUint8_t *ClientPrvKey_ptr1,
				                    DxUint16_t ClientPrvKeySize1,
 									DxUint8_t *ClientPrvKey_ptr2,
				                    DxUint16_t ClientPrvKeySize2,				                    
 				                    DxUint8_t *ServerPubKey_ptr1,
				                    DxUint16_t ServerPubKeySize1,
 				                    DxUint8_t *ServerPubKey_ptr2,
				                    DxUint16_t ServerPubKeySize2,				                    				                    
				                    DxUint8_t *Prime_ptr,
				                    DxUint16_t PrimeSize,
				                    CRYS_DH_DerivationFunc_Mode DerFunc_mode,
				                    DxUint8_t *SecretKeyData_ptr,
				                    DxUint16_t SecretKeyDataSizeBitsNeeded);

/**
   \brief CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data implements the DH ANSI X9.42 standard deriving a hybrid secret key 
   			from 2 public-private pair of keys using the Derivation function based on ASN1
*/
#define CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data(UserPubKey_ptr,PrimeData_ptr,hashFunc,ClientPrvKey_ptr1,ClientPrvKeySize1,ClientPrvKey_ptr2,ClientPrvKeySize2,ServerPubKey_ptr1,ServerPubKeySize1,ServerPubKey_ptr2,ServerPubKeySize2,Prime_ptr,PrimeSize,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)\
		_DX_DH_X942_Hybrid_GetSecretData(UserPubKey_ptr,PrimeData_ptr,hashFunc,ClientPrvKey_ptr1,ClientPrvKeySize1,ClientPrvKey_ptr2,ClientPrvKeySize2,ServerPubKey_ptr1,ServerPubKeySize1,ServerPubKey_ptr2,ServerPubKeySize2,Prime_ptr,PrimeSize,CRYS_DH_ASN1_Der_mode,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)
				                    			  
/**
   \brief CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data implements the DH ANSI X9.42 standard deriving a hybrid secret key 
   			from 2 public-private pair of keys using the Derivation function based on conctenation using SHA-1 Hash
*/				                    			  
#define CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data(UserPubKey_ptr,PrimeData_ptr,hashFunc,ClientPrvKey_ptr1,ClientPrvKeySize1,ClientPrvKey_ptr2,ClientPrvKeySize2,ServerPubKey_ptr1,ServerPubKeySize1,ServerPubKey_ptr2,ServerPubKeySize2,Prime_ptr,PrimeSize,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)\
        _DX_DH_X942_Hybrid_GetSecretData(UserPubKey_ptr,PrimeData_ptr,hashFunc,ClientPrvKey_ptr1,ClientPrvKeySize1,ClientPrvKey_ptr2,ClientPrvKeySize2,ServerPubKey_ptr1,ServerPubKeySize1,ServerPubKey_ptr2,ServerPubKeySize2,Prime_ptr,PrimeSize,CRYS_DH_Concat_Der_mode,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)
                             
                             
#ifdef __cplusplus
}
#endif

#endif
